
/************************************************************************************//**
 * @file		test_demo_CAN.cpp
 * @brief		Demo application for HD CAN Module
 * @date		2013-09-24
 * @copyright	Hatteland Display AS
 ***************************************************************************************/
 
#include "hdj2534Handler.h"
#include <string.h>
#include <iostream>
#include <iomanip>
#include <string>
#include <fstream>
//#include "hdj2534.h"

#define FAKE_CHANNEL_ID		123
#define FAKE_DEVICE_ID		0
#define FAKE_PTR			0
#define FAKE_FLAGS			0xFFFFFF
#define PERIODIC_MSG_NUM 	15
#define FILTER_NUM			15
#define PASSTHRU_MSG_NUM	256
#define ID_BYTES_NUM		4
#define INITIAL_MSG_ID		0xFFFFFF
#define INITIAL_FILTER_ID	0xFFFFFF

using namespace std;
using namespace J2534;

/** @brief J2534 PassThru function name strings */
const char *passThruFuncNames[16] =
{
	"PassThruOpen()",
	"PassThruClose()",
	"PassThruConnect()",
	"PassThruDisconnect()",
	"PassThruReadMsgs()",
	"PassThruWriteMsgs()",
	"PassThruStartPeriodicMsg()",
	"PassThruStopPeriodicMsg()",
	"PassThruStartMsgFilter()",
	"PassThruStopMsgFilter()",
	"PassThruSetProgrammingVoltage()",
	"PassThruReadVersion()",
	"PassThruGetLastError()",
	"PassThruIoctl()",
	"PassThruGetStatus()",
	"PassThruReset()",
	"\0",
};

/** @brief J2534 PassThru function numbers */
enum passThruFuncNum
{
	NO_FUNCTION = 0,
	PASSTHRU_OPEN,
	PASSTHRU_CLOSE,
	PASSTHRU_CONNECT,
	PASSTHRU_DISCONNECT,
	PASSTHRU_READ_MSG,
	PASSTHRU_WRITE_MSG,
	PASSTHRU_START_PERIODIC_MSG,
	PASSTHRU_STOP_PERIODIC_MSG,
	PASSTHRU_START_MSG_FILTER,
	PASSTHRU_STOP_MSG_FILTER,
	PASSTHRU_SET_PROGRAMMING_VOLTAGE,
	PASSTHRU_READ_VERSION,
	PASSTHRU_GET_LAST_ERROR,
	PASSTHRU_IOCTL,
	PASSTHRU_GET_STATUS,
	PASSTHRU_RESET,
};

static unsigned long deviceId;
static unsigned long channel_1;
static unsigned long channel_2;
static unsigned long filterId[FILTER_NUM];
static unsigned long periodMsgId[PERIODIC_MSG_NUM];
static PASSTHRU_MSG txPassThruMsg[PASSTHRU_MSG_NUM];
static PASSTHRU_MSG rxPassThruMsg[PASSTHRU_MSG_NUM];
static PASSTHRU_MSG filterMaskPassThruMsg;
static PASSTHRU_MSG filterPattPassThruMsg;


static void callPassThruOpen();
static void callPassThruClose();
static void callPassThruConnect();
static void callPassThruDisconnect();
static void callPassThruReadMsgs();
static void callPassThruWriteMsgs();
static void callPassThruStartMsgFilter();
static void callPassThruStopMsgFilter();


HINSTANCE dllLib;

PassThruOpen_t PassThruOpen;
PassThruClose_t PassThruClose;
PassThruConnect_t PassThruConnect;
PassThruDisconnect_t PassThruDisconnect;
PassThruReadMsgs_t PassThruReadMsgs;
PassThruWriteMsgs_t PassThruWriteMsgs;
PassThruGetLastError_t PassThruGetLastError;
PassThruStartPeriodicMsg_t PassThruStartPeriodicMsg;
PassThruStopPeriodicMsg_t PassThruStopPeriodicMsg;
PassThruStartMsgFilter_t PassThruStartMsgFilter;
PassThruStopMsgFilter_t PassThruStopMsgFilter;
PassThruIoctl_t PassThruIoctl;
PassThruSetProgrammingVoltage_t PassThruSetProgrammingVoltage;
PassThruReadVersion_t PassThruReadVersion;
PassThruGetStatus_t PassThruGetStatus;
PassThruReset_t PassThruReset;
_J2534_ERROR_CODE err;


bool freeLib()
{
	if (NULL != dllLib)
	{
		if (FreeLibrary(dllLib) == NULL)
		{
			return false;
		}
		dllLib = NULL;
	}
	return true;
}

bool loadLib()
{
#ifdef _WIN64
	dllLib = LoadLibrary(TEXT("HDJ2534x64.dll"));
#else
	dllLib = LoadLibrary(TEXT("HDJ2534x86.dll"));
#endif
	if (dllLib == NULL) 
	{
		cout << "Unable to load DLL!" << endl;
		return false;
	}

	PassThruOpen = (PassThruOpen_t)GetProcAddress(dllLib, "PassThruOpen");
	PassThruClose = (PassThruClose_t)GetProcAddress(dllLib, "PassThruClose");
	PassThruConnect = (PassThruConnect_t)GetProcAddress(dllLib, "PassThruConnect");
	PassThruDisconnect = (PassThruDisconnect_t)GetProcAddress(dllLib, "PassThruDisconnect");
	PassThruReadMsgs = (PassThruReadMsgs_t)GetProcAddress(dllLib, "PassThruReadMsgs");
	PassThruWriteMsgs = (PassThruWriteMsgs_t)GetProcAddress(dllLib, "PassThruWriteMsgs");
	PassThruGetLastError = (PassThruGetLastError_t)GetProcAddress(dllLib, "PassThruGetLastError");
	PassThruStartPeriodicMsg = (PassThruStartPeriodicMsg_t)GetProcAddress(dllLib, "PassThruStartPeriodicMsg");
	PassThruStopPeriodicMsg = (PassThruStopPeriodicMsg_t)GetProcAddress(dllLib, "PassThruStopPeriodicMsg");
	PassThruStartMsgFilter = (PassThruStartMsgFilter_t)GetProcAddress(dllLib, "PassThruStartMsgFilter");
	PassThruStopMsgFilter = (PassThruStopMsgFilter_t)GetProcAddress(dllLib, "PassThruStopMsgFilter");
	PassThruIoctl = (PassThruIoctl_t)GetProcAddress(dllLib,"PassThruIoctl");
	PassThruSetProgrammingVoltage = (PassThruSetProgrammingVoltage_t)GetProcAddress(dllLib,"PassThruSetProgrammingVoltage");
	PassThruReadVersion = (PassThruReadVersion_t)GetProcAddress(dllLib,"PassThruReadVersion");
	PassThruGetStatus = (PassThruGetStatus_t)GetProcAddress(dllLib,"PassThruGetStatus");
	PassThruReset = (PassThruReset_t)GetProcAddress(dllLib,"PassThruReset");

	if ((PassThruOpen == NULL) || 
		(PassThruClose == NULL) || 
		(PassThruConnect == NULL) || 
		(PassThruDisconnect == NULL) || 
		(PassThruReadMsgs == NULL) || 
		(PassThruWriteMsgs == NULL) || 
		(PassThruGetLastError == NULL) || 
		(PassThruStartPeriodicMsg == NULL) || 
		(PassThruStopPeriodicMsg == NULL) || 
		(PassThruStartMsgFilter == NULL) || 
		(PassThruStopMsgFilter == NULL) ||
		(PassThruReset == NULL))		
	{
		cout << "Unable to find DLL function!" << endl;
		if (false == freeLib())
		{
			cout << "Unable to free DLL!" << endl;
		}
		return false;
	}
	return true;
}

/************************************************************************************//**
 * @brief Function to Open device
 ***************************************************************************************/
static void callPassThruOpen()
{
	PassThruOpen(0, &deviceId);
}

/************************************************************************************//**
 * @brief Function to Close Device
 ***************************************************************************************/
static void callPassThruClose()
{
	PassThruClose(deviceId);
}

/************************************************************************************//**
 * @brief Function to Connect Channel
 ***************************************************************************************/
static void callPassThruConnect()
{
	int baudSel;
	int protoSel;
	int chanSel;
	int devSel;
	int flagSel;
	unsigned long deviceIdToUse;
	unsigned long baudrate;
	unsigned long flags;

	//"Enter protocol (0-J1850VPW, 1-J1850PWM, 2-ISO9141, 3-ISO14230, 4-CAN, 5-ISO15765, "
	//		"6-SCI_A_ENGINE, 7-SCI_A_TRANS, 8-SCI_B_ENGINE, 9-SCI_B_TRANS): ";
	//Select CAN protocol
	protoSel = 4;
	//cout << "Enter baudrate (0-125kbps, 1-250kbps, 2-500kbps, 3-1Mbps, 4-invalid baudrate): ";
	baudSel = 125000;

	flags = PHYSICAL_CHANNEL;

	deviceIdToUse = deviceId;
	
	PassThruConnect(deviceIdToUse, static_cast<J2534_PROTOCOL>(protoSel + 1), flags, baudrate, &channel_1);

}

/************************************************************************************//**
 * @brief Function to disconnect channel
 ***************************************************************************************/
static void callPassThruDisconnect()
{
	PassThruDisconnect(channel_1);
}

/************************************************************************************//**
 * @brief Function to Read Message
 ***************************************************************************************/
static void callPassThruReadMsgs()
{
	unsigned long msgNum = 1;
	unsigned long *msgNumPtr = &msgNum;
	unsigned long timeout = 0;
	unsigned int id = 0x10;
	int chanSel;
	int msgPtrSel;
	int msgNumPtrSel;
	J2534_ERROR_CODE err;
	PASSTHRU_MSG *rxPassThruMsgPtr = rxPassThruMsg;
	J2534_RxStatus rxStatus;

	err = PassThruReadMsgs(channel_1, rxPassThruMsgPtr, msgNumPtr, timeout);
	
	if (err == STATUS_NOERROR)
	{
		if (msgNum != 0)
		{
			for (unsigned int i = 0; i < msgNum; i++)
			{
				id = rxPassThruMsg[i].Data[0] << 24;
				id |= rxPassThruMsg[i].Data[1] << 16;
				id |= rxPassThruMsg[i].Data[2] << 8;
				id |= rxPassThruMsg[i].Data[3] << 0;
				cout << hex << uppercase;
				cout << "ID:" << setfill('0') << setw(ID_BYTES_NUM * 2) << id << "\t";
				cout << "DLC:" << setfill('0') << setw(2) << (rxPassThruMsg[i].DataSize - ID_BYTES_NUM) << "\t";
				cout << "MSG:";
				for (unsigned int j = ID_BYTES_NUM; j < rxPassThruMsg[i].DataSize; j++)
				{
					cout << setfill('0') << setw(2) << (int)rxPassThruMsg[i].Data[j] << " ";
				}
				cout << dec;
				cout << "\t";
				cout << "TYPE:";
				rxStatus.value = rxPassThruMsg[i].RxStatus;
				if (rxStatus.bits.Can29BitId == 0)
				{
					cout << "STD" << endl;
				}
				else
				{
					cout << "EXT" << endl;
				}
			}
		}
		else
		{
			cout << "No message has been received" << endl;
		}
	}
}

/************************************************************************************//**
 * @brief Function to Write Messages
 ***************************************************************************************/
static void callPassThruWriteMsgs()
{
	unsigned int id = 0x10;
	unsigned long msgNum = 1;
	unsigned long *msgNumPtr = &msgNum;
	unsigned int dlc = 1;
	J2534_ConnectFlags txFlags;
	unsigned long timeout = 0;;
	int addrSel;
	int chanSel;
	int msgPtrSel;
	int msgNumPtrSel;
	int protoSel;
	PASSTHRU_MSG *txPassThruMsgPtr = txPassThruMsg;

	//"Enter protocol (0-J1850VPW, 1-J1850PWM, 2-ISO9141, 3-ISO14230, 4-CAN, 5-ISO15765, "
	//		"6-SCI_A_ENGINE, 7-SCI_A_TRANS, 8-SCI_B_ENGINE, 9-SCI_B_TRANS): ";
	protoSel = 4;
	
	//Set message Data
	txPassThruMsg[0].Data[ID_BYTES_NUM + 0] = static_cast<unsigned char>(0x12);
	
	//Standard message length
	txFlags.bits.Can29BitId = 0;


	txPassThruMsg[0].TxFlags = txFlags.value;
	txPassThruMsg[0].DataSize = dlc + ID_BYTES_NUM;
	txPassThruMsg[0].ProtocolID = static_cast<J2534_PROTOCOL>(protoSel + 1);
	for (unsigned int i = 0; i < msgNum; i++)
	{
		txPassThruMsg[i] = txPassThruMsg[0];
		txPassThruMsg[i].Data[0] = static_cast<unsigned char>(id >> 24);
		txPassThruMsg[i].Data[1] = static_cast<unsigned char>(id >> 16);
		txPassThruMsg[i].Data[2] = static_cast<unsigned char>(id >> 8);
		txPassThruMsg[i].Data[3] = static_cast<unsigned char>(id >> 0);
		id++;
	}
	//Send Message via Channel_1
	PassThruWriteMsgs(channel_1, txPassThruMsgPtr, msgNumPtr, timeout);

}

/************************************************************************************//**
 * @brief Function to start msg filter
 ***************************************************************************************/
static void callPassThruStartMsgFilter()
{
	int protoSel;
	int addrSel;
	int chanSel;
	int filterIdSel = 0;
	int filterTypeSel = PASS_FILTER;
	unsigned int maskId = 0;
	unsigned int pattId = 0;
	int msgPtrSel;
	PASSTHRU_MSG *filterMaskPassThruMsgPtr = &filterMaskPassThruMsg;
	PASSTHRU_MSG *filterPattPassThruMsgPtr = &filterPattPassThruMsg;
	J2534_ConnectFlags filterFlags;
	J2534_FILTER filterType;

	
	memset((void *)&filterMaskPassThruMsg, 0, sizeof filterMaskPassThruMsg);
	memset((void *)&filterPattPassThruMsg, 0, sizeof filterPattPassThruMsg);

	//Select CAN protocol
	//"Enter protocol (0-J1850VPW, 1-J1850PWM, 2-ISO9141, 3-ISO14230, 4-CAN, 5-ISO15765, "
	//		"6-SCI_A_ENGINE, 7-SCI_A_TRANS, 8-SCI_B_ENGINE, 9-SCI_B_TRANS): ";
	protoSel = 4;
	filterMaskPassThruMsg.ProtocolID = static_cast<J2534_PROTOCOL>(protoSel + 1);
	filterPattPassThruMsg.ProtocolID = static_cast<J2534_PROTOCOL>(protoSel + 1);


	filterMaskPassThruMsg.DataSize = ID_BYTES_NUM;

	filterPattPassThruMsg.DataSize = ID_BYTES_NUM;

	for (unsigned int i = 0; i < 4; i++)
	{
		filterMaskPassThruMsg.Data[ID_BYTES_NUM + i] = 0;
	}

	for (unsigned int i = 0; i < 4; i++)
	{
		filterPattPassThruMsg.Data[ID_BYTES_NUM + i] = 0;
	}
	

	filterMaskPassThruMsg.Data[0] = static_cast<unsigned char>(maskId >> 24);
	filterMaskPassThruMsg.Data[1] = static_cast<unsigned char>(maskId >> 16);
	filterMaskPassThruMsg.Data[2] = static_cast<unsigned char>(maskId >> 8);
	filterMaskPassThruMsg.Data[3] = static_cast<unsigned char>(maskId >> 0);

	filterPattPassThruMsg.Data[0] = static_cast<unsigned char>(pattId >> 24);
	filterPattPassThruMsg.Data[1] = static_cast<unsigned char>(pattId >> 16);
	filterPattPassThruMsg.Data[2] = static_cast<unsigned char>(pattId >> 8);
	filterPattPassThruMsg.Data[3] = static_cast<unsigned char>(pattId >> 0);
	
	//Set Standard ID(0/Standard;1/Extended)
	filterFlags.bits.Can29BitId = 0;

	filterMaskPassThruMsg.TxFlags = filterFlags.value;

	filterFlags.bits.Can29BitId = 0;

	filterPattPassThruMsg.TxFlags = filterFlags.value;


	PassThruStartMsgFilter(channel_1, filterType, filterMaskPassThruMsgPtr, filterPattPassThruMsgPtr, NULL, &filterId[filterIdSel]);

}

/************************************************************************************//**
 * @brief Function to stop message filter
 ***************************************************************************************/
static void callPassThruStopMsgFilter()
{
	int filterIdSel = 0;
	J2534_ERROR_CODE err;

	err = PassThruStopMsgFilter(channel_1, filterId[filterIdSel]);

	if (STATUS_NOERROR == err)
	{
		filterId[filterIdSel] = INITIAL_FILTER_ID;
	}
}

/************************************************************************************//**
 * @brief Function to do software reset
 ***************************************************************************************/
static void callPassThruReset()
{	
	PassThruReset();
}

int main(void)
{
	//Load Library
	if(!loadLib())
	{
		cout << "HDJ2534.dll - load error!" << endl;
		return 0;
	}

	//Open Device
	callPassThruOpen();
	
	//Connect Channel_1 CAN 125000bps
	callPassThruConnect();
	
	//Soft Reset the module
	callPassThruReset();

	//Re-Open Device
	callPassThruOpen();
	
	//Connect Channel_1 CAN 125000bps
	callPassThruConnect();
	
	//Write a single message
	callPassThruWriteMsgs();
	
	//Set Read Message filter 0
	callPassThruStartMsgFilter();
	
	//Read one single message
	callPassThruReadMsgs();
	
	//Delete Read Message Filter 0
	callPassThruStartMsgFilter();
	
	//Disconnect Channel_1
	callPassThruDisconnect();
	
	//Close Device
	callPassThruClose();
	
	freeLib();
	
	return 1;
}




